/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 * This file is part of the LibreOffice project.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 */

#pragma once

#include <string_view>
#include <unoidl/unoidl.hxx>
#include <codemaker/typemanager.hxx>
#include <rtl/string.hxx>

/**
 * RustProducer generates Rust-side opaque pointer wrappers.
 *
 * This is the Rust-side generator in the opaque pointer approach.
 * It creates Rust structs that use void* pointers and call extern "C"
 * bridge functions generated by the C-side (CppProducer).
 *
 * Architecture:
 * - Rust structs with opaque void* pointers
 * - Safe RAII wrappers with Drop trait
 * - extern "C" function declarations matching C-side bridge
 * - All parameters/returns are void* for maximum simplicity
 */
class RustProducer
{
public:
    /**
     * Constructor
     * @param outputDir Directory for generated Rust files
     * @param verbose Enable verbose output
     * @param dryRun Don't actually write files
     * @param typeManager UNO type manager for accessing type information
     */
    RustProducer(std::string_view outputDir, bool verbose, bool dryRun,
                 const rtl::Reference<TypeManager>& typeManager);

    /**
     * Generate Rust opaque enum wrapper
     * Creates simple enum with conversion functions via extern "C"
     */
    void produceEnum(std::string_view name, const rtl::Reference<unoidl::EnumTypeEntity>& entity);

    /**
     * Generate Rust opaque struct wrapper
     * Creates struct with void* pointer and field accessor methods
     */
    void produceStruct(std::string_view name,
                       const rtl::Reference<unoidl::PlainStructTypeEntity>& entity);

    /**
     * Generate Rust opaque interface wrapper
     * Creates struct with void* pointer and method wrappers
     */
    void produceInterface(std::string_view name,
                          const rtl::Reference<unoidl::InterfaceTypeEntity>& entity);

    /**
     * Generate Rust opaque service wrapper
     * Creates service creation wrapper methods
     */
    void produceService(std::string_view name,
                        const rtl::Reference<unoidl::SingleInterfaceBasedServiceEntity>& entity);

private:
    // Configuration
    OString m_outputDir;
    bool m_verbose;
    bool m_dryRun;
    rtl::Reference<TypeManager> m_typeManager;

    // Helper functions
    static std::string_view splitName(std::string_view name);
    static OString getRustFunctionName(std::string_view methodName);
    static OString getRustTypeName(std::string_view unoName);
    OString getRustWrapperTypeName(std::u16string_view unoType) const;
    OString getRustReturnType(std::u16string_view unoType) const;
    OString
    getRustParameterType(std::u16string_view unoType,
                         unoidl::InterfaceTypeEntity::Method::Parameter::Direction direction) const;
    OString getRustExternParameterType(
        std::u16string_view unoType,
        unoidl::InterfaceTypeEntity::Method::Parameter::Direction direction) const;

    // Struct getter/setter type functions
    OString getRustStructGetterReturnType(std::u16string_view unoType) const;
    OString getRustStructSetterParameterType(std::u16string_view unoType) const;
    OString getRustStructExternGetterReturnType(std::u16string_view unoType) const;
    OString getRustStructExternSetterParameterType(std::u16string_view unoType) const;

    OString convertRustParameterForFFICall(
        std::u16string_view unoType, std::string_view paramName,
        unoidl::InterfaceTypeEntity::Method::Parameter::Direction direction) const;
    OString resolveTypedef(std::u16string_view unoType) const;

    // Type classification helpers
    bool isUnoInterface(std::u16string_view typeName) const;
    bool isUnoStruct(std::u16string_view typeName) const;
    bool isUnoEnum(std::u16string_view typeName) const;

    // Enum generation helpers (following CppProducer pattern)
    static void generateEnumDefinition(class RustFile& file, std::string_view name,
                                       const rtl::Reference<unoidl::EnumTypeEntity>& entity);
    static void generateEnumImplementation(class RustFile& file, std::string_view name);
    static void generateEnumExternDeclarations(class RustFile& file, std::string_view name);

    // Struct generation helpers (following CppProducer pattern)
    static void
    generateStructDefinition(class RustFile& file, std::string_view name,
                             const rtl::Reference<unoidl::PlainStructTypeEntity>& entity);
    void generateStructImplementation(class RustFile& file, std::string_view name,
                                      const rtl::Reference<unoidl::PlainStructTypeEntity>& entity);
    static void generateStructConstructor(class RustFile& file,
                                          std::string_view externFunctionPrefix);
    static void generateStructFromPtr(class RustFile& file, std::string_view externFunctionPrefix);
    static void generateStructAsPtr(class RustFile& file);
    void generateStructAccessors(class RustFile& file,
                                 const rtl::Reference<unoidl::PlainStructTypeEntity>& entity,
                                 std::string_view externFunctionPrefix);
    void generateStructMemberGetter(class RustFile& file, std::string_view memberName,
                                    std::u16string_view memberType,
                                    std::string_view externFunctionPrefix);
    void generateStructMemberSetter(class RustFile& file, std::string_view memberName,
                                    std::u16string_view memberType,
                                    std::string_view externFunctionPrefix);
    static void
    generateStructDropTrait(class RustFile& file, std::string_view name,
                            const rtl::Reference<unoidl::PlainStructTypeEntity>& entity);
    void
    generateStructExternDeclarations(class RustFile& file, std::string_view name,
                                     const rtl::Reference<unoidl::PlainStructTypeEntity>& entity);
    static void generateStructBasicExternDeclarations(class RustFile& file,
                                                      std::string_view externFunctionPrefix);
    void generateStructMemberExternDeclarations(
        class RustFile& file, const rtl::Reference<unoidl::PlainStructTypeEntity>& entity,
        std::string_view externFunctionPrefix);

    // Interface generation helpers (following CppProducer pattern)
    void generateInterfaceWrapper(class RustFile& file, std::string_view name,
                                  const rtl::Reference<unoidl::InterfaceTypeEntity>& entity);
    void
    generateInterfaceExternDeclarations(class RustFile& file, std::string_view name,
                                        const rtl::Reference<unoidl::InterfaceTypeEntity>& entity);

    void generateInterfaceImplementation(class RustFile& file, std::string_view typeName,
                                         std::string_view externFunctionPrefix,
                                         const rtl::Reference<unoidl::InterfaceTypeEntity>& entity);
    static void generateInterfaceConstructor(class RustFile& file,
                                             std::string_view externFunctionPrefix);
    static void generateInterfaceFromPtr(class RustFile& file,
                                         std::string_view externFunctionPrefix);
    static void generateInterfaceAsPtr(class RustFile& file);
    static void
    generateInterfaceValidityCheck(class RustFile& file, std::string_view externFunctionPrefix,
                                   const rtl::Reference<unoidl::InterfaceTypeEntity>& entity);
    void generateInterfaceMethodWrappers(class RustFile& file,
                                         std::string_view externFunctionPrefix,
                                         const rtl::Reference<unoidl::InterfaceTypeEntity>& entity);
    static void generateInterfaceDropTrait(class RustFile& file, std::string_view typeName,
                                           std::string_view externFunctionPrefix);
    static void generateInterfaceThreadSafety(class RustFile& file, std::string_view typeName);

    // Service generation helpers (following CppProducer pattern)
    static void generateServiceDefinition(
        class RustFile& file, std::string_view name,
        const rtl::Reference<unoidl::SingleInterfaceBasedServiceEntity>& entity);
    static void generateServiceImplementation(
        class RustFile& file, std::string_view name,
        const rtl::Reference<unoidl::SingleInterfaceBasedServiceEntity>& entity);
    static void generateServiceCreateMethod(
        class RustFile& file, std::string_view name,
        const rtl::Reference<unoidl::SingleInterfaceBasedServiceEntity>& entity);
    static OString generateServiceInterfaceModulePath(std::u16string_view interfaceType);
    static void generateServiceExternDeclarations(
        class RustFile& file, std::string_view name,
        const rtl::Reference<unoidl::SingleInterfaceBasedServiceEntity>& entity);

    // Type casting helpers
    static bool generateTypeCastReturn(class RustFile& file, std::string_view resolvedType);
};

/* vim:set shiftwidth=4 softtabstop=4 expandtab: */
